package jcircus.util;

import java.io.File;
import java.io.FileWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;
import jcircus.environment.ProcChanUseEnv;
import jcircus.environment.ChanInfoEnv;
import jcircus.environment.Environment;

import jcircus.environment.ProcChanEnv;
import jcircus.environment.NameTypeEnv;
import jcircus.environment.TypeList;
import jcircus.exceptions.InvalidFormatCommException;
import jcircus.exceptions.InvalidParameterException;
import jcircus.exceptions.InvalidSubTypeException;
import jcircus.exceptions.NoChannelMSEnvAnnotationException;
import jcircus.exceptions.NoCircusTypeAnnotationException;
import jcircus.exceptions.NoIdCircusProcessAnnException;
import jcircus.exceptions.NoNameTypeAnnotationException;
import jcircus.exceptions.NoProcChanEnvAnnotationException;
import jcircus.exceptions.NoSignatureAnnException;
import jcircus.exceptions.JCircusException;
import jcircus.util.annotations.HideOkAnn;
import jcircus.util.annotations.IdCircusProcessAnn;
import net.sourceforge.czt.base.ast.ListTerm;
import net.sourceforge.czt.base.ast.TermA;
import net.sourceforge.czt.base.impl.ListTermImpl;
import net.sourceforge.czt.circus.ast.Action2;
import net.sourceforge.czt.circus.ast.CircusAction;
import net.sourceforge.czt.circus.ast.Communication;
import net.sourceforge.czt.circus.ast.DotField;
import net.sourceforge.czt.circus.ast.ExtChoiceAction;
import net.sourceforge.czt.circus.ast.Field;
import net.sourceforge.czt.circus.ast.HideProcess;
import net.sourceforge.czt.circus.ast.InputField;
import net.sourceforge.czt.circus.ast.IntChoiceAction;
import net.sourceforge.czt.circus.ast.OutputField;
import net.sourceforge.czt.z.ast.Expr;
import net.sourceforge.czt.z.ast.Name;
import net.sourceforge.czt.z.ast.ProdExpr;
import net.sourceforge.czt.z.ast.RefExpr;
import net.sourceforge.czt.z.ast.RefName;
import net.sourceforge.czt.z.ast.SchExpr;
import net.sourceforge.czt.z.ast.SchText;
import net.sourceforge.czt.z.ast.Signature;
import net.sourceforge.czt.z.util.Factory;
import net.sourceforge.czt.z.ast.ConstDecl;
import net.sourceforge.czt.z.ast.Decl;
import net.sourceforge.czt.z.ast.DeclName;
import net.sourceforge.czt.z.ast.NameTypePair;
import net.sourceforge.czt.z.ast.VarDecl;
import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.Velocity;


/**
 * Util.java
 *
 * Contains utility methods.
 *
 * @author Angela Freitas
 *
 */
public class Util {
    
     public static Factory factory_ = new Factory();
    
    /** **************************************************************************************************************
     * Communication
     * ***************************************************************************************************************
     */
    
    /**
     * Classifies a communication according to its use (ChanUse - 
     * Input, Output or Undefined).
     */
    public static ChanUse getChanUseClassification(Communication communication)
            throws InvalidFormatCommException {
        
        ChanUse r;
        
        RefName chanName = communication.getChanName();
        ListTerm fields = communication.getChanFields();
        
        Field field;
        
        if (fields.size() == 0) {
            // Single sync channel
            r = ChanUse.Undefined;
            
        } else {
            if (fields.size() == 1) {
                field = (Field) fields.get(0);
                
            } else {
                
                for (int i = 0; i < fields.size() - 2; i++) {
                    field = (Field) fields.get(i);
                    
                    if (!(field instanceof DotField))
                        throw new InvalidFormatCommException();
                }
                
                // Gets the last parameter
                field = (Field) fields.get(fields.size() - 1);
            }
            
            if (field instanceof InputField)
                r = ChanUse.Input;
            else if (field instanceof OutputField)
                r = ChanUse.Output;
            else if (field instanceof DotField)
                r = ChanUse.Undefined;
            else
                throw new InvalidSubTypeException(field.getClass());
        }
        
        return r;
    }
    
    /**
     * Classifies a communication regarding synchronisation (ChanSync - S or C).
     */
    public static ChanSync getChanSyncClassification(Communication communication) 
            throws InvalidFormatCommException {
        
        ChanSync r;
        
        RefName chanName = communication.getChanName();
        ListTerm fields = communication.getChanFields();
        
        Field field;
        
        if (fields.size() == 0) {
            // Single sync channel
            r = ChanSync.S;
            
        } else {
            if (fields.size() == 1) {
                
                field = (Field) fields.get(0);
                
            } else {
                
                for (int i = 0; i < fields.size() - 2; i++) {
                    field = (Field) fields.get(i);
                    
                    if (!(field instanceof DotField))
                        throw new InvalidFormatCommException();
                }
                
                // Gets the last parameter
                field = (Field) fields.get(fields.size() - 1);
            }
            
            if (field instanceof InputField)
                r = ChanSync.C;
            else if (field instanceof OutputField)
                r = ChanSync.C;
            else if (field instanceof DotField)
                r = ChanSync.S;
            else
                throw new InvalidSubTypeException(field.getClass());
        }
        
        return r;
    }
    
    /** **************************************************************************************************************
     * RenameVars
     * ***************************************************************************************************************
     */
    
    /**
     * Rename variables in Java code.
     */
    public static String renameVars(String code, String newVar, String oldVar) 
            throws Exception {
        
        List newVars = new ArrayList();
        newVars.add(newVar);
        List oldVars = new ArrayList();
        oldVars.add(oldVar);
        
        return Util.renameVars(code, newVars, oldVars);
    }
    
    /**
     * Rename variables in Java code.
     */
    public static String renameVars(String code, List/*<String or Name>*/ newVars, 
            List/*<String or Name>*/ oldVars) throws InvalidSubTypeException {
        
        String newCode = "";
        StringTokenizer st = new StringTokenizer(code, ".;,(){} \t\n\r\f", true);
        String token;
        Object object;
        int index;
        
        while (st.hasMoreTokens()) {
            
            token = st.nextToken();
            
            index = contains(oldVars, token);
            
            if (index != -1) {
                
                object = newVars.get(index);
                if (object instanceof String)
                    // visitIntChoiceActionIte, visitMuAction, visitParAction, visitSeqActionIte
                    token = (String) object;
                else if (object instanceof Name)
                    // visitRenameProcess
                    token = ((Name) object).toString();
                else
                    throw new InvalidSubTypeException(object.getClass());
            }
            
            newCode = newCode + token;
        }
        
        return newCode;
    }
    
    /**
     * Returns the position of a string in a list. Returns -1 if it is not
     * in the list.
     */
     private static int contains(List/*<Name or String>*/ list, String token) {
        
        int r = -1;
        
        Object obj;
        
        for (int i = 0; i < list.size(); i++) {
            
            obj = list.get(i);
            if (obj.toString().equals(token)) {
                r = i;
                break;
            }
        }
        return r;        
    }
    
    /** **************************************************************************************************************
     * Annotation
     * ***************************************************************************************************************
     */
    
    /**
     * NameType
     *
     * @param termA
     * @param nameType
     */
    public static void addNameTypeAnn(TermA termA, NameType nameType) {
        
        termA.getAnns().add(nameType);
    }
    
    /**
     * NameType
     *
     * @param termA
     * @param nameType
     */
    public static NameType getAndRemoveNameTypeAnn(TermA termA) 
            throws NoNameTypeAnnotationException {
        
        // Get
        NameType nameType = Util.getNameTypeAnn(termA);
        
        // Remove
        termA.getAnns().remove(nameType);
        
        return nameType;
    }

    /**
     * NameType
     *
     * @param termA
     * @param nameType
     */
    public static NameType getNameTypeAnn(TermA termA) 
            throws NoNameTypeAnnotationException {

        // Get
        NameType nameType = (NameType) termA.getAnn(NameType.class);
        
        // Checks if it is null
        if (nameType == null)
            throw new NoNameTypeAnnotationException();
        
        return nameType;
    }

    /**
     * ChannelMSEnv
     */
    public static void addChannelMSEnvAnn(TermA termA, ChanInfoEnv channelMSEnv) {
        termA.getAnns().add(channelMSEnv);
    }
    
     /*
      * ChannelMSEnv
      */
    public static ChanInfoEnv getChannelMSEnvAnn(TermA termA) 
            throws NoChannelMSEnvAnnotationException {

        // Get
        ChanInfoEnv channelMSEnv = (ChanInfoEnv) termA.getAnn(ChanInfoEnv.class);
        
        // Checks if it is null
        if (channelMSEnv == null)
            throw new NoChannelMSEnvAnnotationException();
        
        return channelMSEnv;
    }
    
    /**
     * CircusType
     */
    public static void addCircusTypeAnn(TermA termA, CircusType circusType) {
        
        termA.getAnns().add(circusType);
    }
    
    /**
     * CircusType
     */
    public static CircusType getCircusTypeAnn(TermA termA) 
            throws NoCircusTypeAnnotationException {

        // Get
        CircusType circusType = (CircusType) termA.getAnn(CircusType.class);
        
        // Checks if it is null
        if (circusType == null)
            throw new NoCircusTypeAnnotationException();
        
        return circusType;
    }

    /**
     * ProcessChannelEnvironment
     */
    public static void addProcChanEnvAnn(TermA termA, ProcChanEnv procChanEnv) {
        termA.getAnns().add(procChanEnv);
    }

    public static ProcChanEnv getProcChanEnvAnn(TermA termA) 
            throws NoProcChanEnvAnnotationException {

        // Get
        ProcChanEnv procChanEnv = (ProcChanEnv) termA.getAnn(ProcChanEnv.class);
        
        // Checks if it is null
        if (procChanEnv == null)
            throw new NoProcChanEnvAnnotationException();
        
        return procChanEnv;
    }
    
    /**
     * ProcessChannelEnvironment
     */
    public static void addNameTypeEnvAnn(TermA termA, NameTypeEnv nameTypeEnv) {
        
        termA.getAnns().add(nameTypeEnv);
    }
    
    /**
     * These are temporary.
     * @ deprecated ?
     * @param termA
     */
    public static void addListTermAnn(TermA termA, ListTerm listTerm) {
        
        termA.getAnns().add(listTerm);
    }
    
    /**
     * @ deprecated ?
     */
    public static ListTerm getListTermAnn(TermA termA) throws Exception {

        // Get
        ListTerm listTerm = (ListTerm) termA.getAnn(ListTerm.class);
        
        // Checks if it is null
        if (listTerm == null)
            throw new Exception("No annotation");
        
        return listTerm;
    }
    
    /**
     * IdCircusProcess
     */
    public static void addIdCircusProcessAnn(TermA termA, IdCircusProcessAnn ann) {
        
        termA.getAnns().add(ann);
    }
    
    public static IdCircusProcessAnn getIdCircusProcessAnn(TermA termA) 
            throws NoIdCircusProcessAnnException {

        // Get
        IdCircusProcessAnn ann = (IdCircusProcessAnn) 
            termA.getAnn(IdCircusProcessAnn.class);
        
        // Checks if it is null
        if (ann == null)
            throw new NoIdCircusProcessAnnException();
        
        return ann;
    }

    /**
     * Signature.
     */
    public static void addSignatureAnn(RefName refName, Signature ann) {
        
        refName.getAnns().add(ann);
    }
    
    public static Signature getSignatureAnn(RefName refName) 
            throws NoSignatureAnnException { 

        // Get
        Signature ann = (Signature) refName.getAnn(Signature.class);
        
        // Checks if it is null
        if (ann == null)
            throw new NoSignatureAnnException();  
        
        return ann;
    }
    
    /**
     * Adds a HideOkAnn.
     */
    public static void addHideOkAnn (HideProcess hideProcess) {
        hideProcess.getAnns().add(new HideOkAnn());
    }
    
    /**
     * Returns the HideOkAnn; null if there is none.
     */
    public static HideOkAnn getHideOkAnn (HideProcess hideProcess) {
        return (HideOkAnn) hideProcess.getAnn(HideOkAnn.class);
    }
    
    /*
     * Leo's method.
     *
    publicvoid addLocAnn(TermA termA, int line, int column)
    {
      LocAnn locAnn = fCF.createLocAnn(fFileName, line + 1, column + 1);
      LocAnn existingAnn = (LocAnn) termA.getAnn(LocAnn.class);
      if (existingAnn != null) {
        assert locAnn.equals(existingAnn);
        return;
      }
      termA.getAnns().add(locAnn);
    }
     */
    
    /** **************************************************************************************************************
     * External and Internal Choice
     * ***************************************************************************************************************
     */
    
    /**
     * This method return all the actions that take part in an internal or
     * external choice in the first level. Basically it transforms the binary
     * operation into an n-ary operation.
     *
     */
    public static List<CircusAction> getActions(CircusAction circusAction, int operation) {
        
        List result = new ArrayList();
        
        if ((operation == Constants.OP_EXTCHOICE && circusAction instanceof ExtChoiceAction)
                 || (operation == Constants.OP_INTCHOICE && circusAction instanceof IntChoiceAction)) {
            
            result.addAll(getActions(
                    ((Action2) circusAction).getLeftAction(), operation));
            result.addAll(getActions(
                    ((Action2) circusAction).getRightAction(), operation));
            
        } else if (operation == Constants.OP_EXTCHOICE || 
                operation == Constants.OP_INTCHOICE) {
            
            // Base case
            result.add(circusAction);
            
        } else
            throw new InvalidParameterException("Operation");
        
        return result;
    }
    
    /** **************************************************************************************************************
     * Synchronization Channels
     * ***************************************************************************************************************
     */
    
    /**
     * Returns a string of '[]', as many times as the dimension of the channel.
     */
    public static String arrayDim(CircusType channelType, ChanSync sc, int gap) {
        
        String r = "";
        int dim = Util.dim(channelType, sc);
        
        return arrayDim(dim - gap);
    }
    
    /**
     * Returns a string of '[]', 'dim' times.
     */
    public static String arrayDim(int dim) {
        
        String r = "";
        
        for (int i = 0; i < dim; i++) {
            r = r + "[]";
        }
        
        return r;
    }
    
    /**
     * Returns the dimension of the channel. Takes into account:
     * - the number of generic parameters
     * - the number of Fields
     * - if it is S or C
     */
    public static int dim(CircusType channelType, ChanSync sc) {
        
        int dim;
        
        List/*<Name>*/ genericParameters = channelType.getGenericParameters();
        List/*<Expr>*/ types;
        
        if (channelType.isSyncType()) {
            // Single synchronization
            types = new ArrayList();
            types.add(Util.createSyncExpression());
            
        } else {
            
            Expr typeExpression = channelType.getExpression();
            if (typeExpression instanceof ProdExpr) {
                types = ((ProdExpr) typeExpression).getExpr();
            } else {
                types = new ArrayList();
                types.add(typeExpression);
            }
        }
        
        return dim(genericParameters, types, sc);
    }
    
    /**
     * Returns the dimension of the channel. Takes into account:
     * - the number of generic parameters (genTypes)
     * - the number of Fields  (types)
     * - if it is S or C (sc)
     */
    public static int dim(List/*Name*/ genTypes, List/*Expr*/ types, ChanSync sc) {
        
        int dim;
        
        if (types.size() == 1 && types.get(0).equals(Util.createSyncExpression())) {
            dim = 0;
        } else if (sc.equals(ChanSync.C)) {
            dim = types.size() - 1;
        } else {
            dim = types.size();
        }
        
        if (genTypes != null) {
            // It is a generic type
            dim = dim + genTypes.size();
        }
        
        return dim;
    }
    
    /**
     *
     */
    public static String instArray(CircusType circusTypeChannel, 
            ChanSync syncType, TypeList typeList) {
        
        String code = "";
        List/*<Name>*/ genPars = new ArrayList();
        List/*<Expr>*/ types;
        
        if (circusTypeChannel.isGeneric()) {
            genPars = circusTypeChannel.getGenericParameters();
        }
        
        if (circusTypeChannel.isSyncType()) {
            types = new ArrayList();
            types.add(Util.createSyncExpression());
        } else {
            Expr typeExpression = circusTypeChannel.getExpression();
            if (typeExpression instanceof ProdExpr) {
                types = ((ProdExpr) typeExpression).getExpr();
            } else {
                types = new ArrayList();
                types.add(typeExpression);
            }
            
        }
        
        List listOfTypes = new ArrayList();
        Iterator iterator = typeList.iterator();
        
        while (iterator.hasNext()) {
            
            CircusType circusType = (CircusType) iterator.next();
            listOfTypes.add(circusType.getJavaCircusTypeName());
        }
        
        return code;
    }
    
    /**
     * Metodo replace: simplificar, considerar lista de tipos 
     * podendo conter apenas free types.
     *
     * genTypes: List of Name
     * types: List of String
     * typeList: List of String
     *
     */
    public static String instArray(List/*<Name>*/ genTypes, 
            List/*<Expr>*/ types, ChanSync sc, List/*String*/ typeList) {
        
        String code = "";
        int dim = dim(genTypes, types, sc);
        
        if (genTypes.size() > 0) {
            code = "new Any2OneChannel " + arrayDim(dim) + "{ " +
                    genericInst(genTypes, types, sc, typeList, typeList) + " }";
        } else {
            code = instArraySync(types, sc);
        }
        
        return code;
    }
    
    /**
     *
     */
    public static String genericInst(List/*<Name>*/ genTypes, List/*<String>*/ types, 
            ChanSync sc, List/*<String>*/ typeList, List/*<String>*/ typeList2) {
        
        String code = "";
        
        Name genPar = (Name) genTypes.get(0);
        String type = (String) typeList2.get(0);
        
        if (typeList2.size() == 1) {
            
            code = instArray(genTypes.subList(1, genTypes.size()),
                    replace(genPar, type, types), sc, typeList);
        } else {
            
            code = instArray(genTypes.subList(1, genTypes.size()), replace(genPar, type, types), sc, typeList)
            + ", " +
                    genericInst(genTypes, types, sc, typeList, typeList2.subList(1, typeList2.size()));
        }
        
        return code;
    }
    
    /**
     *
     */
    public static String instArraySync(List/*<String>*/ types, ChanSync sc) {
        
        String code = "";
        
        int dim = dim(new ArrayList(), types, sc);
        String type = (String) types.get(0);
        
        if (types.size() == 1) {
            
            code = baseCase(type, sc);
        } else {
            
            int n = 0; //... TODO
            
            code = " new Any2OneChannel " + arrayDim(dim) + "{ " +
                    typeInstSync(types, sc, n) + "} ";
        }
        
        return code;
    }
    
    /**
     * Invokes the method instArray for the other types
     *
     *
     * @param types
     * @param sc
     * @param n
     * @return
     */
    public static String typeInstSync(List/*String*/ types, ChanSync sc, int n) {
        
        String code = "";
        
        if (n == 1) {
            
            code = instArraySync(types.subList(1, types.size()), sc);
        } else {
            code = instArraySync(types.subList(1, types.size()), sc) + ", " +
                    typeInstSync(types, sc, n - 1);
        }
        
        return code;
    }
    
    /**
     * Replace every reference to 'genPar' by 't' in 'types'.
     *
     * @param genPar
     * @param t
     * @param types
     * @return
     */
    public static List replace(Name genPar, String t, List/*<String>*/ types) {
        
        List newList = new ArrayList();
        Iterator iterator = types.iterator();
        
        while(iterator.hasNext()) {
            newList.add(new String(t));
        }
        
        return newList;
    }
    
    /**
     * Returns Java Code, the instantiation of a Channel.
     *
     * sc = C -> instantiates a single Any2OneChannel
     * sc = S -> instantiates an array of channels with the number of elements 
     * equals to the number of possible values of the type
     *
     * @param type
     * @param sc
     * @return
     */
    public static String baseCase(String type, ChanSync sc) {
        
        String code = "";
        
        // TODO: Initialize 'n' properly
        int n = 0;
        
        if (sc.equals(ChanSync.C))
            code = "new Any2OneChannel()";
        else
            code = "Any2OneChannel.create(" + n + ")";
        
        return code;
    }
    
    /**
     * Usado em dim(), instArray(), CircusType.createSyncType()
     * @return
     */
    public static Expr createSyncExpression() {
        
        Factory factory = new Factory();
        return factory.createRefExpr(factory.createRefName(CircusType.SYNC_CHANNEL));
        
    }
    
    /**
     * Returns an empty ListTerm.
     */
    public static List createEmptyList() {
        return new ListTermImpl();
    }
    
    /** **************************************************************************************************************
     * Circus Type
     * ***************************************************************************************************************
     */
    
    /**
     *
     */
    public static CircusType getLastTypeChannel(CircusType circusTypeChannel, 
            ListTerm genActuals) throws InvalidParameterException {
        
        CircusType r;
        Expr lastTypeExpr;
        Expr typeExpression = circusTypeChannel.getExpression();

        if (typeExpression instanceof ProdExpr) {

            // Builds a Circus type with the last expr
            List expressions = ((ProdExpr) typeExpression).getExpr();
            lastTypeExpr = (Expr) expressions.get(expressions.size()-1);

        } else {
            // Returns the same type that was received as parameter
            lastTypeExpr = circusTypeChannel.getExpression();
        }

        if (circusTypeChannel.isGeneric()) {
            
            // Case of a generic type
            List genParams = circusTypeChannel.getGenericParameters();
            
            if (lastTypeExpr instanceof RefExpr) {
                RefName name = ((RefExpr) lastTypeExpr).getRefName();
                
                // Checks if the type is a generic param, that is, if it 
                // appears in the array of generic parameters
                for (int i = 0; i < genParams.size(); i++) {
                    DeclName genParam = (DeclName) genParams.get(i);
                    if (genParam.toString().equals(name.toString())) {
                        // If this is the case, then gets the correspondent
                        // type expression in the array of actuals
                        lastTypeExpr = (Expr) genActuals.get(i);
                        break;
                    }
                }
            } else {
                // For the moment, I am dealing only with Free Types.
                throw new InvalidParameterException("");
            }
        }
        
        // make a new type with the type expression of the last variable
        r = new CircusType(lastTypeExpr);
        
        return r;
    }
         
    /** **************************************************************************************************************
     * Template
     * ***************************************************************************************************************
     */
     
    /**
     * Returns Java code contained in a Velocity template.
     *
     * @param   String      templName   Name of the template
     * @param   Hashtable   ht          Maps a String (parameter) to another String.
     */
    public static String getCodeFromTemplate(String templName, Hashtable ht) {
        
        String result = "";
        
        try {
            VelocityContext velocityContext;
            Template template = null;

            // Velocity code
            velocityContext = new VelocityContext();
            Velocity.init();

            Iterator it = ht.keySet().iterator();

            while (it.hasNext()) {

                String key = (String) it.next();
                String code = (String) ht.get(key);

                velocityContext.put(key, code);
            }

            template = Velocity.getTemplate(templName);

            StringWriter stringWriter = new StringWriter();

            template.merge(velocityContext, stringWriter);
            stringWriter.close();
            result = stringWriter.toString();
        
        } catch (Throwable t) {
            throw new JCircusException("Velocity exception", t);
        }
        
        return result;
    }

    /**
     * Creates a file with the code in a Velocity template.
     *
     * @param   String      fileName    Name of the file
     * @param   String      templName   Name of the template
     * @param   Hashtable   ht          Maps a String (parameter) to another String.
     */  
    public static String createFileFromTemplate(String fileName, 
            String templName, Hashtable ht) {
        
        String result = "";
        
        try {
            VelocityContext velocityContext;
            Template template = null;

            File file = new File(fileName);
            FileWriter fileWriter = new FileWriter(fileName);

            // Velocity code
            velocityContext = new VelocityContext();
            Velocity.init();

            Iterator it = ht.keySet().iterator();

            while (it.hasNext()) {

                String key = (String) it.next();
                String code = (String) ht.get(key);

                velocityContext.put(key, code);
            }

            template = Velocity.getTemplate(templName);

            template.merge(velocityContext, fileWriter);
            fileWriter.close();
        
        } catch (Throwable t) {
            throw new JCircusException("Velocity exception", t);
        }
        
        return result;
    }

    /** **************************************************************************************************************
     * Code
     * ***************************************************************************************************************
     */
    
    /**
     * This method is called for the constructor of a class.
     */
    public static String instChannelSimple(String channelName, ProcChanUseEnv chanMsEnv, 
            Integer procId, boolean isMain) {
        
        String code = "";
        
        List ids = resolveUndefinedChannels(chanMsEnv, isMain);
        
        // Declaration of chanInfo
        code += "\n" + Constants.CLS_CHINFO + " chanInfo_" + channelName + " = new " + Constants.CLS_CHINFO + "();";
        
        // Initialization of chanInfo
        for (int i = 0; i < ids.size(); i++) {
            Integer subProcId = (Integer) ids.get(i);
            code = code + "\nchanInfo_" + channelName + 
                    ".put(new Integer(" + subProcId + "), new Integer(" + i + "));";
        }

        // Instantiation of GeneralChannel
        code += "\nthis." + channelName + " = new GeneralChannel(" +
                "new Any2OneChannel(), chanInfo_" + channelName + 
                ", new Integer(" + procId.intValue() + "));";

        return code;
    }
    
    /**
     * Returns a list containing the names of the processes, in the
     * position corresponding to the multisync id.
     * 
     * Basically, tries to find any Output channel in chanMSEnv, which
     * will be given the position zero. Assigns any position to the
     * other channels.
     *
     * Maybe it would be better that this method returned a hashtable instead
     * of a list, because there can be many writers in the case where there is
     * not a multi-synchronization... So, all these writers should have id zero.
     *
     */
    public static List resolveUndefinedChannels(ProcChanUseEnv chanMSEnv, 
            boolean isMain) {
     
        // Tratar diferentemente os casos em que ha interface grafica e os que 
        // nao ha interface grafica. No caso em que ha interface grafica 
        // (!chanMSEnv.isSync()) devera ja adicionar o id da interface grafica (-1)
        // como reader ou como writer.
        
        // para decidir como eu vou fazer aqui dar uma olhada no metodo 
        // getChanUseGuiNotSyncChannel de chanMsEnv.
        // este metodo  chamado em processCallCode para decidir quem sera o READER
        // e quem sera o WRITER (o processo ou a gui).

        // Creates an array
        Integer[] array;
        
        boolean writerDefined = false;
        int index = 0;
        
        if (!chanMSEnv.isSync()) {
            // There is no synchronization involving the channel
            // Will take part in gui, in case isMain is true
            
            if (!isMain) {
                array = new Integer[chanMSEnv.size()];
                // If this is being called from a constructor of a process (that is,
                // the channel is hidden, then there is no concern about the gui.
                // We just add all the processes in chanMSEnv to any position
                // of the array.
                Iterator it = chanMSEnv.iteratorKeys();
                
                while(it.hasNext()) {
                    Integer procId = (Integer) it.next();
                    array[index] = procId;
                    index++;
                }
                
            } else {
                array = new Integer[chanMSEnv.size()+1]; // +1 for the gui
                // Checks if the process will be input or output
                ChanUse chanUseNotSync = chanMSEnv.getChanUseGuiNotSyncChannel();

                if(chanUseNotSync.equals(ChanUse.Input)) {
                    // gui will be output
                    array[0] = new Integer(-1);

                    // iterates over the array, inserting the other positions
                    index = 1;                
                    Iterator it = chanMSEnv.iteratorKeys();
                    while(it.hasNext()) {
                        Integer procId = (Integer) it.next();
                        ChanUse chanUse = chanMSEnv.get(procId);
                        array[index] = procId;
                        index++;
                    }

                } else {
                    // gui will be input
                    array[1] = new Integer(-1);
                    index = 2;
                    writerDefined = false;
                    Iterator it = chanMSEnv.iteratorKeys();
                    while(it.hasNext()) {
                        Integer procId = (Integer) it.next();
                        ChanUse chanUse = chanMSEnv.get(procId);

                        // The first Undefined found goes to the writer position.
                        // There will be a bug if there is more than one Undefined
                        // because then only one of them will go to the writer
                        // position, therefore the others will try to read the 
                        // channel instead of write.
                        if (chanUse.equals(ChanUse.Undefined) && !writerDefined) {
                            array[0] = procId;
                            writerDefined = true;
                        } else {
                            if (index == chanMSEnv.size() + 1) {
                                // Reached the end and the writer had not been defined
                                array[0] = procId;
                            }
                        }
                    }
                }
            }
        } else {
            array = new Integer[chanMSEnv.size()];
            // There is simple, or multiple synchronization involving the channel
            index = 0;
            writerDefined = false;
            Iterator it = chanMSEnv.iteratorKeys();
            while(it.hasNext()) {
                Integer procId = (Integer) it.next();
                ChanUse chanUse = chanMSEnv.get(procId);

                if (chanUse.equals(ChanUse.Output) && !writerDefined) {

                    // Output channel
                    writerDefined = true;
                    array[0] = procId;

                } else {
                    // Input channel or Undefined channel
                    index++;

                    // This is the last element and there is any writer in the array
                    if (index == array.length)
                        index = 0;

                    array[index] = procId;
                }
            }
        }
        
        // Transforms the array into a list
        List list = new ArrayList();
        for (int i=0; i<array.length; i++)
            list.add(array[i]);
        
        return list;
    }    
    
    /** **************************************************************************************************************
     * Other
     * ***************************************************************************************************************
     */

    public static void print(String st, String cl, String method) {
        
        boolean debug = true;
        
        if (debug)
            System.out.println(":: " + cl + ", " + method + ": " + st);
    }
    
    
}
